####################################################################################################
#
#                                           EXTERNAL LIBRARIES
#
####################################################################################################

import networkx     as     nx
from   progress_bar import ProgressBar
import os
import sys

####################################################################################################
#
#                                           FUNCTIONS
#
####################################################################################################

def readArg( arg_values , arg_name , arg_nb , verbosity, *default ):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   A function for reading arguments of a script
	|
	[Arguments]-----------------------------------------------------------------
	[Returned values]-----------------------------------------------------------
	[Example]-------------------------------------------------------------------
	|---------------------------------------------------------------------------
	"""
	if len(arg_values) > 1:
		arg_pos = 1
		while ( arg_pos < ( len( arg_values ) -1 ) ) and arg_values[ arg_pos ] != arg_name:
			arg_pos += 1
		if ( arg_values[ arg_pos ] == arg_name ):
			if arg_nb == 1 :
				return True
			else:
				return arg_values[ arg_pos + 1 ]
		else:
			if default[0] == 'none':
				return 1
			else:
				if verbosity:
					sys.stdout.write('Fail to read %s argument value.\n'%(arg_name))
					sys.stdout.write('Default value \'%s\' used.\n'%(str(default[0])))
				return default[0]
	else:
		if default[0] == 'none':
			return 1
		else:
			if verbosity:
				sys.stdout.write('Fail to read %s argument value.\n'%(arg_name))
				sys.stdout.write('Default value \'%s\' used.\n'%(str(default[0])))
			return default[0]

def range_length(r):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the length of a range
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   r : a list of 2 integers setting the initial and final limits of the
	|       range
	|
	[Returned values]-----------------------------------------------------------
	|
	|   an integer = ( sup - inf + 1 )
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> range_length( [5,10] )
	|   6
	|
	|---------------------------------------------------------------------------
	"""
	if len(r) == 0:
		return 0
	else:
		return r[1]-r[0]+1


def ranges_lengths(listofranges):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the length of an union of ranges
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   listofranges : a list of sub-lists of 2 integers setting the initial
	|                  and final limits of the ranges
	|
	[Returned values]-----------------------------------------------------------
	|
	|   an integer = sum over the ranges of ( sup - inf + 1 )
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> ranges_lengths( [ [5,10], [12,20]] )
	|   15
	|
	|---------------------------------------------------------------------------
	"""
	cummulativelength = 0
	for r in listofranges:
		cummulativelength += range_length(r)
	return cummulativelength


def common_range( range1 , range2 ):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the limits of tow ranges intersection
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   range1, range2 : a list of 2 integers setting the initial and final
	|                    limits of the range
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list of 2 integers setting the sup and inf limits of a range
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> common_range( [5,10], [8,13] )
	|   [8,10]
	|
	|---------------------------------------------------------------------------
	"""
	if ((len(range1) != 0) and (len(range2) != 0)):
		commonrange = [ max(range1[0],range2[0]) , min(range1[1],range2[1]) ]
		#if commonrange[1] > commonrange[0]:
		if commonrange[1] >= commonrange[0]: # modif le 9/2/2016
			return commonrange
		else:
			return []
	else:
		return []


def common_ranges( listrange1 , listrange2 ):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the limits of the ranges resulting from the intersection of 2
	|   lists of ranges
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   listrange1, listrange2 : a list of sub-lists of 2 integers setting the
	|                            initial and final limits of the ranges
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list of sub-lists of 2 integers setting the sup and inf limits of 
	|   the ranges
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> common_ranges( [[5,10], [8,13]], [[1,6],[12,20]] )
	|   [[5,6],[12,13]]
	|
	|---------------------------------------------------------------------------
	"""
	commonranges = []
	for range1 in listrange1:
		for range2 in listrange2:
			commonranges.append( common_range( range1 , range2) )
	for r in commonranges:
		if r == []:
			commonranges.remove([])
	return commonranges


def common_ranges_lengths( listrange1 , listrange2 ):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the length of the intersection of 2 lists of ranges
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   listrange1, listrange2 : a list of sub-lists of 2 integers setting the
	|                            initial and final limits of the ranges
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a integer
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> common_ranges_lengths( [[5,10], [8,13]], [[1,6],[12,20]] )
	|   4
	|
	|---------------------------------------------------------------------------
	"""
	return ranges_lengths( common_ranges( listrange1 , listrange2 ) )


def flatten_tuple(t) :
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Take a list of lists or a tuple of tuples and return a one dimension
	|   vector of the sorted union of their values 
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   t : a list of sub-lists or a tuple of tuples
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> flatten_tuple( ( ( 5 , 10 ) , ( 5 , 13 ) ) )
	|   [5, 10, 13]
	|
	|---------------------------------------------------------------------------
	"""
	d = {}
	for i in t :
		for j in i :
			d[j] = 1
	l = d.keys()
	l.sort()
	return l


def test_ternary_similarity( ali1 , ali2 , sim ) :
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Test the ternary similarity between two pair of items.
	|
	|   Evaluate the overlaping lentgh on the common item of the correspondance
	|   map of the two set of ranges.
	|   Test gives True if the overlaping length is over sim percent of the 
	|   length of the shortest map
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   ali1 , ali2 : two set of aligned_items
	|   sim         : a float comprize between 0. and 1.
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a boolena
	|
	|[Example]------------------------------------------------------------------
	|
	|
	|---------------------------------------------------------------------------
	"""
	min_len = sim * float( min( ali1.align_length() , ali2.align_length() ) )
	if   ali1.it1 == ali2.it1 :
		over_len = common_ranges_lengths( ali1.blocks_it1 , ali2.blocks_it1 )
	elif ali1.it1 == ali2.it2 :
		over_len = common_ranges_lengths( ali1.blocks_it1 , ali2.blocks_it2 )
	elif ali1.it2 == ali2.it1 :
		over_len = common_ranges_lengths( ali1.blocks_it2 , ali2.blocks_it1 )
	elif ali1.it2 == ali2.it2 :
		over_len = common_ranges_lengths( ali1.blocks_it2 , ali2.blocks_it2 )
	return ( over_len >= min_len )  # modif 9/2/2016
	#return ( over_len > min_len )

def xcombinations(items, n):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns all possible combination n items taken from a list 
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   items : a list or a tuple
	|   n     : an iteger - the number of combined items
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list of lists of dimension n
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> list(xcombinations((1,'e','fr'),2))
	|   [[1, 'e'], [1, 'fr'], ['e', 1], ['e', 'fr'], ['fr', 1], ['fr', 'e']]
	|
	|---------------------------------------------------------------------------
	"""
	if n==0: yield []
	else:
		for i in xrange(len(items)):
			for cc in xcombinations(items[:i]+items[i+1:],n-1):
				yield [items[i]]+cc

def xuniqueCombinations(items, n):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns all possible combination n items taken from a list where
	|   combinations are uniques
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   items : a list or a tuple
	|   n     : an iteger - the number of combined items
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list of lists of dimension n
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> list(xuniqueCombinations((1,'e','fr'),2))
	|   [[1, 'e'], [1, 'fr'], ['e', 'fr']]
	|
	|---------------------------------------------------------------------------
	"""
	if n==0: yield []
	else:
		for i in xrange(len(items)):
			for cc in xuniqueCombinations(items[i+1:],n-1):
				yield [items[i]]+cc


def mean(t):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the mean value over a table of number
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   t : a list or a tuple
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a float
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> mean( [ 1 , 2 , 2 ] )
	|   2.5
	|
	|---------------------------------------------------------------------------
	"""
	m = 0
	for i in t:
		m += i
	return float(m)/float(len(t))


def write_graph( filename , G , type , dico , verbosity ):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Write in output file a graph
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   filename  <string>          : a file path
	|
	|   G         <networkx.Graph()>: a graph
	|
	|   type      <integer>         : 1 -> each G node is an item
	|                                 2 -> each G node is a pair of items
	|
	|   dico      <dict>            : keys : internal item ID
	|                                 vals : user item name
	|
	|   verbosity <boolean>         :
	|
	[Returned values]-----------------------------------------------------------
	|
	|   void
	|
	|[Example]------------------------------------------------------------------
	|
	|---------------------------------------------------------------------------
	"""
	fod = open(filename,'w')
	if verbosity :
		sys.stdout.write('\n[  Start ] writing graph to %s\n'%filename)
		sys.stdout.flush()
	
	if type == 1 :
		if verbosity :
			iter = 0
			prog = ProgressBar( iter , G.number_of_edges(), 70 , mode='fixed', char='-')
		for n1,n2 in G.edges():
			fod.write( '%s\t%s\n'%( dico[n1] ,  dico[n2] ) )
			if verbosity :
				# progress-bar
				iter += 1
				prog.increment_amount()
				sys.stderr.write( '%s\r'%(prog) )
				sys.stderr.flush()
		if verbosity :
			sys.stderr.write( '\n'%(prog) )
			sys.stderr.flush()
			iter = 0
			prog = ProgressBar( iter , G.number_of_nodes(), 70 , mode='fixed', char='-')
		for n in G.nodes():
			if G.degree(n) == 0 :
				fod.write( '%s\n'%( dico[n] ) )
			if verbosity :
				# progress-bar
				iter += 1
				prog.increment_amount()
				sys.stderr.write( '%s\r'%(prog) )
				sys.stderr.flush()
		if verbosity :
			sys.stderr.write( '\n'%(prog) )
			sys.stderr.flush()
	elif type == 2 :
		if verbosity :
			iter = 0
			prog = ProgressBar( iter , G.number_of_edges(), 70 , mode='fixed', char='-')
		for n1,n2 in G.edges():
			fod.write( '( %s - %s )\t( %s - %s )\n'%( dico[n1[0]] , \
			                                          dico[n1[1]] , \
			                                          dico[n2[0]] , \
			                                          dico[n2[1]] ) )
			if verbosity :
				# progress-bar
				iter += 1
				prog.increment_amount()
				sys.stderr.write( '%s\r'%(prog) )
				sys.stderr.flush()
		if verbosity :
			sys.stderr.write( '\n'%(prog) ); sys.stderr.flush()
			iter = 0
			prog = ProgressBar( iter , G.number_of_nodes(), 70 , mode='fixed', char='-')
		for n in G.nodes():
			if G.degree(n) == 0 :
				fod.write( '( %s - %s )\n'%( dico[n[0]] ,dico[n[1]] ) )
			if verbosity :
				# progress-bar
				iter += 1
				prog.increment_amount()
				sys.stderr.write( '%s\r'%(prog) )
				sys.stderr.flush()
	if verbosity :
		sys.stderr.write( '\n'%(prog) )
		sys.stderr.flush()
		sys.stdout.write('[    End ] writing graph to %s\n'%filename)
		sys.stdout.flush()
	
	fod.close()

def flatten_uniq(t):
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Realize the union list of a list of list
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   t  <list_of_list> 
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> flatten_uniq( [ [ 1 , 2 , 2 ] , [ 1 , 3 ] , [ 4 ] ])
	|   [ 1 , 2 , 3 , 4 ]
	|
	|---------------------------------------------------------------------------
	"""
	d = {}
	for i in t :
		for j in i:
			d[j]=1
	tfu = d.keys()
	tfu.sort()
	return tfu



def neighbors_edges( G , edge ) :
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the list of adjacent edges of one G edge
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   G     < networkx.Graph() > 
	|   edge  <  list or tuple   > an edge of G
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a list of edges
	|
	|[Example]------------------------------------------------------------------
	|
	|---------------------------------------------------------------------------
	"""
	nei_e    = {}
	n1       = edge[ 0 ]
	n2       = edge[ 1 ]
	n1_neighbors = G.neighbors( n1 ); n1_neighbors.remove( n2 );
	n1_neighbors = map( lambda n : sorted_tuple( [ n1 , n ] ) , n1_neighbors )
	n2_neighbors = G.neighbors( n2 ); n2_neighbors.remove( n1 )
	n2_neighbors = map( lambda n : sorted_tuple( [ n2 , n ] ) , n2_neighbors )
	for e in n1_neighbors :
		nei_e[e] = 1
	
	for e in n2_neighbors :
		nei_e[e] = 1
	
	return nei_e.keys()

def sorted_tuple( pair ) :
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the sorted tuple of a list or a tuple
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   pair  <list or tuple> 
	|
	[Returned values]-----------------------------------------------------------
	|
	|   a tuple
	|
	|[Example]------------------------------------------------------------------
	|
	|   >>> sorted_tuple( [ 7 , 2 , 5 ] )
	|   [ 2 , 5 , 7 ]
	|
	|---------------------------------------------------------------------------
	"""
	p = list( pair )
	p.sort()
	return tuple( p )

def number_of_fails_at_ternary_similarity_test( Align_dict , G , edge , Sim_Threshold ) :
	"""
	[Goal]----------------------------------------------------------------------
	|
	|   Returns the number of fails at the centered ternary similarity test
	|   for an edge of G
	|
	[Arguments]-----------------------------------------------------------------
	|
	|   Align_dict  <  dict              > The dictionnary of the alignments.
	|                                      Each alignment corresponds to an edge
	|                                      of G
	|                                      keys   : a pair of items (an edge of G)
	|                                      values : an aligned_items class object
	|   G             < networkx.Graph() > 
	|   edge          <   list or tuple  > a pair of items (a alignment or an
	|                                      edge of G)
	|   Sim_Threshold <   list or tuple  > The threshold of the centered
	|                                      ternary similarity test
	|
	[Returned values]-----------------------------------------------------------
	|
	|   an integer
	|
	|[Example]------------------------------------------------------------------
	|
	|---------------------------------------------------------------------------
	"""
	nb_of_mark = 0
	adj_list = []
	ali_ref  = Align_dict[ sorted_tuple( edge ) ]
	for e in neighbors_edges( G , edge ) :
		ali_test = Align_dict[ e ]
		if not test_ternary_similarity( ali_ref , ali_test , Sim_Threshold ) :
			nb_of_mark += 1
	
	return nb_of_mark

