# -*- coding: utf8 -*-

import networkx as nx
import Itemset
import string
import operator

class RelationalItemsetException(Exception):
	"""
	Exception levée par la classe RealationalItemset
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)

class RelationalItemset :
	"""
	Un itemset relationnel décrit le jeux de données:
		[-] _items		<Items> donne la liste explicite des attributs décrits par les itemsets
		[-] _tIDs maintient un ordre sur les transactions. Par défaut il s'agit de l'ordre dans
			lequel les transaction ont été ajoutées.
		[-] transaction	<dict> donne la description de la partie transactionnelle du jeu
			de données.
			-*- clef: un tID
			-*- val: l'itemset associé à la transaction
		[-] _graph		<nx.Graph> donne la descritpion de la partie relationnelle du jeu
			de donnée. Une transaction (tID) est un noeuds de ce graphe. Une arrête lie 2
		[-] un _titre	

	Rque: Idealement la liste des transactions (tIDs) devrait être maintenue par la liste des
	noeuds du graphe, comme par la liste des clefs du dictionnaire transaction. Ce n'est pas le
	cas. Pour l'instant quand on ajoute une transaction, on ajoute un noeud et ce n'est pas
	reciproque. Par ailleurs, la délétion des transactions n'est pas gérée...
	"""
	def __init__(self, items): 
		"""
		Le constructeur prend la référence d'un Items en parametre( cela fixe le codage et le
		type d'itemset que l'on peut associer à une transaction).
		"""
		if not isinstance( items, Itemset.Items) :
			raise RelationalItemsetException( "%s doit être de type Items\n"%str(items))
		self._items = items
		self._tIDs = []
		self._transaction = dict()
		self._graph = nx.Graph()
		self._titre = ""

	def get_graph(self):
		return self._graph

	def get_items(self):
		return self._items
	
	def add_transaction( self, tID, itemset) :
		"""
		Ajoute une transaction au jeu de données. Une tansaction est consitutée:
			[-] d'un tID unique (identifiant ou nom d'objet)
			[-] d'un Itemset. L'itemset doit être codé avec le même jeu d'étiquettes (Items)
				que celui qui est associé à self.
		"""
		# unicité du tID
		if self._transaction.has_key( tID) :
			raise RelationalItemsetException( "%s deja present dans la table de transaction\n"%str(tID))
		# l'itemset est bien un Itemset
		if not isinstance(itemset, Itemset.Itemset) :
			raise RelationalItemsetException( "%s doit être de type Itemset\n"%str(itemset))
		# L'ItemsetRelationnel et l'Itemset ont le même jeux d'attribut (Items)
		if not itemset.get_items() == self._items :
			raise RelationalItemsetException( "Le jeu d'attribut relatif à l'itemset diffère du jeu d'attribut de référence de litemset relationnel\n ")
		# Ajoute une transaction à la table des transactions
		self._transaction[ tID] = itemset
		# Ajoute un noeud au graphe correspondant à la transaction.
		self._graph.add_node( tID)
		# Ajout le tID dans la liste des transactions
		self._tIDs.append(tID)
	
	def number_of_tIDs(self):
		return len(self._tIDs)
	
	def get_tIDs(self) :
		return self._tIDs
	
	def itemset( self, tID):
		"""
		Renvoie l'itemset associé à la transaction tID 
		"""
		return self._transaction[ tID]
	
	def has_tID( self, tID) :
		"""
		Renvoie Vrai si tID est l'identifiant d'une transaction de l'itemset relationnel
		"""
		return self._transaction.has_key(tID)

	def add_relation( self, tID_1, tID_2) :
		"""
		Ajoute une relation sous forme d'un lien dans le graphe entre 2 transactions.
		Le lien n'est ajouté que si les 2 identifiants correspondent à des tID de transactions
		déjà référencées dans la table self._transaction
		"""
		if not self.has_tID(tID_1) :
			raise RelationalItemsetException("La relation ne peut être ajoutée car %s n'est pas l'identifiant d'une transaction\n"%str(tID_r1))
		if not self.has_tID(tID_2) :
			raise RelationalItemsetException("La relation ne peut être ajoutée car %s n'est pas l'identifiant d'une transaction\n"%str(tID_2))
		self._graph.add_edge( tID_1, tID_2)

	def sort(self) :
		"""
		Retourne une copie de self pour laquelle l'ordre des attributs à été modifié
		pour suivre l'ordr lexicographique
		"""
		idx=[i for i,j in  sorted(list(enumerate(self._items)), key=operator.itemgetter(1))]
		sortedRI = RelationalItemset( Itemset.Items([self._items[i] for i in idx]))
		for tID in self._tIDs:
			sortedRI.add_transaction(tID, Itemset.Itemset(self._transaction[tID].sortedValue(idx),sortedRI.get_items() ))
		sortedRI._graph = self._graph
		sortedRI._titre = self._titre + " avec les attributs tries selon l'ordre lexicographique"
		return sortedRI
	
	def cloture( self, *tIDs) :
		clos = Itemset.Itemset(self.get_items().itemset_max_value(), self.get_items())
		for tID in tIDs :
			clos = clos.intersection( self.itemset(tID))
		return clos
	
	def extension(self, itemset) :
		"""
		Renvoie l'extension (la liste des tIDs des transactions) qui presentent le motif
		decrit par l'itemset.
		"""
		ext = []
		for tID,it in self._transaction.iteritems():
			if ( itemset.intersection(it) == itemset ) :
				ext.append( tID)
		return ext
	
	def support(self,itemset) :
		"""
		Renvoie le support (entier) qui présentent (inclusion) le motif decrit par l'itemset
		"""
		sup = 0
		for tID,it in self._transaction.iteritems():
			if ( itemset.intersection(it) == itemset ) :
				sup += 1
		return sup
	
	def subgraph( self, itemset) :
		"""
		Retourne le sous-graphe induit par l'extension du motif decrit par l'itemset.
		"""
		return self._graph.subgraph( self.extension( itemset) )

	def str_rcf( self) :
		"""
		Renvoie la chaine de caractere correspondant a un fichier *.rcf
		"""
		st  = "[Relational Context]\n"
		st += self._titre+"\n"
		st += "[Binary Relation]\n"
		st += "Genere par ProjClos\n"
		st += string.join( self._tIDs, " | ")+"\n"
		st += string.join(self._items, " | ")+"\n"
		#st += str(tIDs)+"\n"
		for tID in self._tIDs :
			st += self._transaction[tID].binStr()+"\n"
		st += "[END Relational Context]\n"
		return st
	
	def __str__(self) :
		st  = self.str_rcf()
		st += "[Graph Relation]\n"
		for u in self._tIDs :
			v = sorted(self._graph.neighbors(u))
			if 0 < len(v) :
				st += "%s %s\n"%(str(u), string.join(v ,","))
		st += "[END Graph Relation]"
		return st
	
if __name__ == '__main__' :
	items = Itemset.Items(["blues","folk","jazz","pop","rock"])
	ri = RelationalItemset(items)
	ri.add_transaction("A", Itemset.Itemset(int(0b01101),items))
	ri.add_transaction("B", Itemset.Itemset(int(0b11101),items))
	ri.add_transaction("C", Itemset.Itemset(int(0b01101),items))
	ri.add_transaction("D", Itemset.Itemset(int(0b01101),items))
	ri.add_transaction("E", Itemset.Itemset(int(0b11001),items))
	ri.add_transaction("F", Itemset.Itemset(int(0b11000),items))
	ri.add_relation('A','B')
	ri.add_relation('A','C')
	ri.add_relation('A','D')
	ri.add_relation('E','F')







