# -*- coding: utf8 -*-
import RelationalItemset
import Itemset
import string
import re

class RIParserException(Exception):
	"""
	Exception levée par la classe RIParser
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)

class RIParser :
	"""
	Propose des méthodes permettant de lire un fichier au format .ri
	"""
	def __init__(self, fi) :
		self._f = fi
		self._titreTMP = None
		self._ri = None
	
	def parse( self ) :
		"""
		Parse un fichier au format ri et renvoie une instance de RelationalItemset.
		"""
		#-------
		self.__parseHeaders("[Relational Context]")
		self._titreTMP = self._f.readline().strip()
		self.__parseHeaders("[Binary Relation]")
		self._f.readline().strip()
		tIds = self.__parsePipeSeparated()
		self._ri = RelationalItemset.RelationalItemset(Itemset.Items(self.__parsePipeSeparated()))
		self._ri.titre= self._titreTMP
		i = 0
		while (i < len(tIds) ):
			self._ri.add_transaction( tIds[i], Itemset.Itemset( int( self.__parseItemset(), 2), self._ri.get_items() ))
			i += 1
		self.__parseHeaders("[END Relational Context]")
		self.__parseHeaders("[Graph Relation]")
		relations = self.__parseRelations()
		while( relations != "end" ) :
			for u,v in relations:
				self._ri.add_relation(u,v)
			relations = self.__parseRelations()
		return self._ri
	
	def __parseRelations(self):
		"""
		Parcours les lignes du fichier *.ri qui décrivent les relations entre transaction
		Chaque relation doit donner lieu a une arete dans le graphe.
		Les relations sont retournees comme une liste de couples.
		"""
		edges = []
		li = self._f.readline()
		if re.search( "^\[END Graph Relation\]", li) != None:
			return "end"
		m = re.match(r"^([\w\-@]+)\s*(.*)", li)
		for o in re.findall(r"([\w\-@]+)\s*,?",m.group(2)):
			edges.append((m.group(1),o))
		return edges

	def __parseHeaders( self, header) :
		"""
		Verifie que la ligne contient le texte "header". Si le texte est absent une
		exception est levee.
		"""
		li = self._f.readline().strip()
		if header != li :
			raise RIParserException("\"%s\" attendu. \"%s\" lu\n"%(header,li))
			
	def __parsePipeSeparated(self) :
		"""
		Parcours les listes de mots ou d'identifiants separes par un pipe (|) et
		renvoie la liste des identifiants trouves.
		Cela sert notament a parser la liste des objets et la listes des identifiants
		des transactions.
		"""
		return re.findall(r"([\w\-@]+)\s*\|*",self._f.readline())

	def __parseItemset( self) :
		"""
		Parcours les lignes decrivant les transactions sous forme booleennes.
		Chaque ligne correspond a une transaction, et est former d'une suite
		de 0 et de 1 en nombre egal au nombre d'"attributs.
		La fonction renvoie la suite de 0 e de 1 dans une chaine de caracteres.
		"""
		li = self._f.readline()
		binStr = string.join(re.findall(r"[0-1]",li),'')
		if len(binStr) != len( self._ri.get_items() ):
			if re.search(r"^\[END Relational Context\]", li) != None :
				raise RIParserException("Le nombre de vecteurs binaires differe du nombre de transactions.")
			else :
				raise RIParserException("Vecteur binaire de longueur %d attendu, mais vecteur binaire de longueur %d trouve."%(len(self._ri.get_items()),len(binStr)))
		return binStr
	

if __name__ == '__main__' :
	f_in = open('data/mougel.ri', 'r')
	parser = RIParser(f_in)
	ri = parser.parse()
	f_in.close()
	print ri
	print ri.sort()



