# -*- coding: utf8 -*-

import string
from Itemset import Items, Itemset

#-------------------------------------------------------------------------------
class ItemsetRulesException(Exception):
	"""
	Exception levée par la classe ItemsetRules
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)


#-------------------------------------------------------------------------------
class ItemsetRules :
	"""
	Cette structure maintien un jeu de regle liant des itemsets de la forme
		cl <-| ( pr_1, pr_2, ..., prN)
		     |-> extension
		la regle associe
			[-] à itemset cl (pour conclusion)
				(°) une liste d'itemset pr_i (pour premisse).
				(°) une extension (scalaire ou liste de litteraux)
	
	Afin de matinenir la cohérence de l'ensemble, tous les itemsets doivent etre definis relativement au meme
	jeu d'Items
			[-] _items <Items> donne la liste explicite des attributs
	"""
	
	def __init__( self, items ) :
		"""
		Le constructeur prend la reference d'un Items en parametre (cela fixe le codage et une
		description explicite ou codée sous forme d'entier de l'itemset.
		"""
		if not isinstance( items, Items) :
			raise ItemsetRulesException( "%s doit être de type Items\n"%str(items))
		self._items = items
		self._dico = dict()
	
	def get_items(self):
		return self._items
	
	def set_premisses( self, it_cl, it_premisses ) :
		"""
		Définie la liste des premisses pour l'itemset de conclusion it_cl a
		l'ensemble des itemsets de la liste it_premisses.
		Si l'itemset it_cl est deja dans la liste de regles, alors l'extension
		est conservee, sinon une regle est ajoute avec une extension None.
		"""
		try :
			self._dico[ it_cl][0] = set(it_premisses)
		except KeyError :
			self._dico[it_cl] = [set(it_premisses), None]
	
	def _add_rule(self, it_cl, it_pr) :
		"""
		Ajoute l'itemset premisse it a la regle de conclusion it_cl.
		Si la conclusion it_cl n'existe pas, alors elle est creee avec l'itemset it pour seule premisse.
		"""
		# on suppose qu'il a ete verifie que l'items de it_cl a deja ete testé comme identique a celui de self
		if not it_pr.get_items() == self._items :
			raise ItemsetRulesException("L'itemset %s a ete defini relativement a un autre jeu d'Items"%str(it))
		try :
			self._dico[it_cl][0].add( it_pr)
		except KeyError :
			self._dico[it_cl] = [set([it_pr]),None]
	
	def add_rule( self, it_cl, *premisses) :
		"""
		Ajoute ou creet une regle dont la conclusion est it_cl et y ajoute les premisses.
		"""
		if not it_cl.get_items() == self._items :
			raise ItemsetRulesException("L'itemset %s a ete defini relativement a un autre jeu d'Items"%str(it_cl))
		for it_pr in premisses :
			self._add_rule(it_cl, it_pr)
	
	def iter_conclusions(self) :
		"""
		Un iterateur sur les conclusions des regles.
		"""
		return self._dico.iterkeys()

	def iter_premisses(self) :
		"""
		Un iterateur sur les premisses des regles.
		"""
		return self._dico.itervalues()
	
	def iter_rules(self):
		"""
		Un iterateur sur les regles (couple conclusion/premisse)
		"""
		return self._dico.iteritems()

	def premisses(self, it_cl):
		"""
		Renvoie les premisses associees a l'intemset de conclusion it_cl
		"""
		try:
			return self._dico[it_cl][0]
		except KeyError:
			raise ItemsetRulesException("Aucune regle associee a la clef %s\n"%str(it_cl))
	
	def set_extension( self, it_cl, support) :
		"""
		Donne une valeur a l'extension d'une regle. Cette extension peut etre un scalaire ou une liste de litteraux
		(ie, d'objets).
		"""
		try:
			if isinstance(support, list):
				self._dico[it_cl][1] = sorted(support)
			else:
				self._dico[it_cl][1] = support
		except KeyError:
			raise ItemsetRulesException("Une regle doit etre associee a la clef %s pour pouvoir y ajouter une extension.\n"%str(it_cl))
	
	def get_extension( self, it_cl) :
		"""
		Retourne la valeur de l'extension d'une regle. Cette extension peut etre un scalaire ou une liste de litteraux
		(ie, d'objets).
		"""
		try:
			return self._dico[it_cl][1]
		except KeyError:
			raise ItemsetRulesException("Aucune regle associee a la clef %s donc pas d'extension.\n"%str(it_cl))
	
	def __str__(self) :
		return string.join( [ str(it_cl)+" ("+str(self.get_extension(it_cl))+") +; ["+string.join([str(e) for e in self.premisses( it_cl)]," ,") + "]" for it_cl in sorted(self._dico.keys(), key=lambda it : len(it.explicite()), reverse=True)], "\n")




#-------------------------------------------------------------------------------
if __name__ == '__main__' :
	ok = True
	items = Items( ["pop", "rock", "folk", "blues", "jazz"])
	# ---------
	clos_gen = ItemsetRules( items)
	clos_gen.add_rule(Itemset(3,items),Itemset(1,items),Itemset(2,items))
	ok &= str(clos_gen)=="{blues, jazz} (None) +; [{jazz} ,{blues}]"
	clos_gen.add_rule(Itemset(6,items),Itemset(1,items),Itemset(2,items))
	ok &= "{blues, jazz} (None) +; [{jazz} ,{blues}]\n{folk, blues} (None) +; [{jazz} ,{blues}]"==str(clos_gen)
	# ---------
	if not ok :
		print "Certain tests ont echoue."
	else :
		print "Tout les tests OK."
#
