# -*- coding: utf8 -*-

import string

#-------------------------------------------------------------------------------
class ItemsException(Exception):
	"""
	Exception levée par la classe Items
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)


#-------------------------------------------------------------------------------
class Items( tuple ) :
	"""
	Décrit l'ensemble des attributs (items) pouvant apparaitre dans une transaction.
	Il s'agit d'une liste ordonnée de littéraux.
	L'ordre est important car il permet de passer d'une description explicite d'un
	itemset donné sous forme d'un sous ensemble des	items, a une descritpion sous 
	forme d'un champs de bits codé sur un entier.
	"""
	def check_binarized_itemset(self, itemset ) :
		"""
		Lève une exception si itemset n'est pas un itemset dérivant de self.
		"""
		if not isinstance( itemset, Itemset ) or not itemset.get_items() == self:
			raise ItemsException( itemset)
			

	def check_has_item( self, explicit_item ) :
		"""
		Lève une exception si l'explicit_item n'est pas un membre de self.
		"""
		if not (explicit_item in self ):
			raise ItemsException( explicit_item )

	def check_explicit_itemset( self, explicit_itemset ) :
		"""
		Lève une exception si l'un des items constituant explicit_itemset n'est pas 
		un membre de self.
		"""
		for item in explicit_itemset :
			self.check_has_item(item)
	
	def binarise( self, explicit_itemset) :
		"""
		Renvoie l'entier codant pour un itemset. Cet entier est a considerer comme un
		champs de bits. Le n-ieme bit du champ vaut 1 si le n-ieme item de self
		est présent dans l' explicit_itemset et 0 sinon.

		Rque:
			Si self est ("pop", "rock", "folk", "blues", "jazz") alors
				[-] "jazz"  est l'item à la position 0
				[-] "blues" est l'item à la position 1
				[-] "pop"   est l'item à la position 4
			Donc l'itemset
				[-] ("jazz")  est codé par 1
				[-] ("blues") est codé par 2
				[-] ("folk")  est codé par 4
				[-] ("blues", "jazz")  est codé par 3

		"""
		self.check_explicit_itemset( explicit_itemset)
		binarized_itemset = 0
		i = 0
		for item in self :
			if item in explicit_itemset :
				binarized_itemset |= ( 1 << ( len(self) - i - 1) )
			i += 1
		return Itemset( binarized_itemset, self)
	
	def itemset_max_value(self) :
		"""
		Renvoie le champs de bit (int) correspondant à l'occurence de tous les attributs
		"""
		return 2**len(self) - 1
	
	def __str__(self) :
		"""
		Défini l'affichage explicite de self
		"""
		return '{'+string.join( self, ", ")+'}'
	
#-------------------------------------------------------------------------------
class ItemsetException(Exception):
	"""
	Exception levée par la classe Itemset
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)


#-------------------------------------------------------------------------------
class Itemset :
	"""
	Un Itemset est décrit par
		[-] _value <int> : un entier qui doit être interprété comme un champs de bits
		                  décrivant un itemset. Le decodage est possible grace a
						  l'ensemble ordonné items qui donne les attributs littéraux
						  associé à chaque bit.
						  Si le n-ieme bit de l'entier val vaut 1 alors l'itemset
						  contient le n-ième atribut de la liste items. Si le bit
						  vaut zero alors il ne contient pas l'attribut.
		[-] _items <Items> donne la liste explicite des attributs
	"""
	
	def __init__( self,val, items ) :
		"""
		Le constructeur prend la reference d'un Items en parametre (cela fixe le codage et une
		description explicite ou codée sous forme d'entier de l'itemset.
		"""
		if not isinstance( items, Items) :
			raise ItemsetException( "%s doit être de type Items\n"%str(items))
		self._items = items
		if isinstance(val, int) or isinstance(val, long):
			if val < 0 or val > self._items.itemset_max_value() :
				raise ItemsetException("la valeur %s de l'itemset est inconsistante avec le jeu d'attributs de reference.\n"%str(val))
			else :
				self._value = val
		elif isinstance(val,tuple) or isinstance(val, list) or isinstance(val, set) :
			self._value = items.binarise(val)
		else:
			raise ItemsetException(val)
	
	def get_items(self) :
		return self._items
	
	def get_value(self) :
		return self._value
	
	def explicite( self) :
		"""
		Decode self pour l'exprimer avec les littéraux de l'ensemble d'attributs
		items
		"""
		explicit_itemset = [] 
		i = 0
		for item in self._items :
			if 1 == ( 1 & self._value >> ( len(self._items) - i - 1 ) ) :
				explicit_itemset.append( item )
			i += 1
		return explicit_itemset

	def intersection( self, *others) :
		"""
		Renvoie l'itemset composé de l'intersection des attributs de l'itemset
		self et des itemset passés en paramètre (il peut y en avoir plusieurs).
		"""
		val = self._value
		for other in others :
			if self._items != other._items:
				raise ItemsetException(other)
			val &= other._value
		return Itemset( val , self._items)
		
	def union( self, *others) :
		"""
		Renvoie l'itemset composé de l'union des attributs de l'itemset self
		et des itemset passés en paramètre (il peut y en avoir plusieurs).
		"""
		val = self._value
		for other in others :
			if self._items != other._items:
				raise ItemsetException(other)
			val |= other._value
		return Itemset( val , self._items)
		
	def sortedValue(self,idx) :
		"""
		Renvoir le champs de bit de self réordonné d'après la liste idx
		"""
		new = 0
		for i,pos in enumerate(reversed(idx)) :
			new |= ( (1 & (self._value >> len(self._items) - pos - 1 )) << i)
		return new
	
	def __eq__(self, other) :
		return (self._value == other._value ) and (self._items == other._items )

	def __hash__(self) :
		return hash(self._value)
	
	def binStr( self ) :
		"""
		Retourne une chaine de 0 et de 1 exprimant le vecteur d'attribut
		sous forme binaire.
		"""
		return string.join(str(bin(self._value))[2:].rjust(len(self._items),'0'),' ')
		
	def __str__( self ) :
		"""
		Renvoie une chaine de caractère décrivant explicitement l'itemset binarisé
		"""
		return '{'+string.join(self.explicite(),", ") + '}'


#-------------------------------------------------------------------------------
if __name__ == '__main__' :
	ok = True
	items = Items( ["pop", "rock", "folk", "blues", "jazz"])
	# ---------
	ok &= items == Items( ["pop", "rock", "folk", "blues", "jazz"])
	# ---------
	try:
		items.check_has_item( "pop" )
		ok &= True
	except ItemsException :
		ok &= False
	# ---------
	try:
		items.check_explicit_itemset( ["pop", "rock"])
		ok &= True
	except ItemsException :
		ok &= False
	# ---------
	try :
		items.check_has_item( "po" )
		ok &= False
	except ItemsException :
		ok &= True
	# ---------
	try :
		items.check_explicit_itemset( ["pop", "ro"])
		ok &= False
	except ItemsException :
		ok &= True
	
	# ---------
	ok &= items.binarise( [ "jazz"] ) == Itemset(1,items)
	ok &= items.binarise( [ "blues"] ) == Itemset(2,items)
	ok &= items.binarise( [ "folk"] ) == Itemset(4,items)
	ok &= items.binarise( [ "blues", "jazz"] ) == Itemset(3,items) 
	ok &= items.binarise( items ) == Itemset(2**(len(items)) - 1, items )
	# ---------
	try :
		items.check_binarized_itemset(Itemset(1, items) ) 
		ok &= True
	except ItemsException :
		ok &= False
	try :
		items.check_binarized_itemset( Itemset(31,items) ) 
		ok &= True
	except ItemsException :
		ok &= False
	try :
		items.check_binarized_itemset( ["pop"] ) 
		ok &= False
	except ItemsException :
		ok &= True
	# ---------
	ok &= str( items) == "{pop, rock, folk, blues, jazz}"
	# ---------
	# ---------
	# ---------
	# ---------
	items = Items( ["pop", "rock", "folk", "blues", "jazz"])
	# ---------
	try:
		Itemset(1, items )
		ok &= True
	except ItemsetException :
		ok &= False
	# ---------
	try:
		Itemset( 31, items)
		ok &= True
	except ItemsetException :
		ok &= False
	# ---------
	try :
		Itemset(32, items)
		ok &= False
	except ItemsetException :
		ok &= True
	# ---------
	try :
		Itemset(-1,items)
		ok &= False
	except ItemsetException :
		ok &= True
	
	# ---------
	ok &= Itemset(1,items).explicite() == ["jazz"]
	ok &= Itemset(2,items).explicite() == ["blues"]
	ok &= Itemset(4,items).explicite() == ["folk"]
	ok &= Itemset(3,items).explicite() == ["blues","jazz"]
	ok &= items.binarise([                            "jazz"]).get_value() == 1
	ok &= items.binarise([                    "blues"       ]).get_value() == 2
	ok &= items.binarise([             "folk"               ]).get_value() == 4
	ok &= items.binarise([                    "blues","jazz"]).get_value() == 3
	ok &= items.binarise([             "folk","blues","jazz"]).get_value() == 7
	# ---------
	ok &= str(Itemset(1, items)) == '{jazz}'
	ok &= str(Itemset(2, items)) == '{blues}'
	ok &= str(Itemset(3, items)) == '{blues, jazz}'
	# ---------
	ok &= Itemset(1, items).intersection(Itemset(1,items))                   == Itemset(1, items)
	ok &= Itemset(4, items).intersection(Itemset(5,items))                   == Itemset(4, items)
	ok &= Itemset(4, items).intersection(Itemset(5,items),Itemset(13,items)) == Itemset(4, items)
	
	ok &= items.binarise(["jazz","folk"]).intersection(items.binarise(["jazz"])) == items.binarise(["jazz"])
	ok &= items.binarise([       "folk"]).intersection(items.binarise(["jazz"])) == items.binarise([])
	ok &= items.binarise(["pop","jazz","folk"]).intersection(items.binarise(["pop","jazz"]), items.binarise(["jazz"])) == items.binarise(["jazz"])

	ok &= Itemset(1, items).union(Itemset(7,items))                          == Itemset(7, items)
	ok &= Itemset(8, items).union(Itemset(7,items))                          == Itemset(15, items)
	ok &= Itemset(1, items).union(Itemset(6,items))                          == Itemset(7, items)
	ok &= Itemset(1, items).union(Itemset(2,items), Itemset(4,items))        == Itemset(7, items)

	ok &= items.binarise(["folk"]).union(items.binarise(["jazz"])) == items.binarise(["folk","jazz"])
	ok &= items.binarise(["folk"]).union(items.binarise(["pop"]),items.binarise(["jazz"])) == items.binarise(["pop","folk","jazz"])
	# ---------
	ok &= items.binarise(["pop","rock","folk","blues","jazz"]).binStr() == "1 1 1 1 1"
	ok &= items.binarise([      "rock","folk","blues","jazz"]).binStr() == "0 1 1 1 1"
	ok &= items.binarise([      "rock","folk","blues"       ]).binStr() == "0 1 1 1 0"
	ok &= items.binarise([      "rock",       "blues"       ]).binStr() == "0 1 0 1 0"
	ok &= items.binarise(["pop",                      "jazz"]).binStr() == "1 0 0 0 1"
	ok &= items.binarise([                            "jazz"]).binStr() == "0 0 0 0 1"
	# ---------
	ok &= Itemset( int(0b01000), items).sortedValue([0,2,3,4,1]) == int(0b00001)
	ok &= Itemset( int(0b01000), items).sortedValue([0,2,3,1,4]) == int(0b00010)
	ok &= Itemset( int(0b01000), items).sortedValue([0,2,1,3,4]) == int(0b00100)
	ok &= Itemset( int(0b01000), items).sortedValue([0,1,2,3,4]) == int(0b01000)
	# ---------
	if not ok :
		print "Certain tests ont echoue."
	else :
		print "Tout les tests OK."
#
