# -*- coding: utf8 -*-
import Itemset
import ItemsetRules
import RelationalItemset
import GrapheDesTriangles
import networkx as nx
import string

class AbstractionControllerException(Exception):
	"""
	Exception levée par la classe AbstractionController
	"""
	def __init__( self, value) :
		self.value = value
	def __str__(self) :
		return repr(self.value)

class AbstractionController :
	"""
	Classe permettant d'appliquer une abstraction sur un jeu de regles. Elle a vocation
	a etre specialisee
	Elle maintient les attributs suivants:

	_rela_itemset <RelationalItemset>
		Donnees de depart ayant servis a calculer les regles Clos/Generateurs

	_clos_gen_ori <ItemsetRules> :
		règles avant application de l'abstraction
			[-] la conclusion est un clos
			[-] les premisses sont des generateurs

	_clos_gen_abs <ItemsetRules> :
		règles après application de l'abstraction
			[-] la conclusion est un clos abstrait
			[-] les premisses sont des generateurs abstraits

	_fusion_clos <ItemsetRules> :
		regles decrivant la fusion des classes d'equivalence durant l'abstraction
			[-] la conclusion clos decrivant la classe après abstraction
			[-] les premisses sont les clos avant abstraction ayant fusionnes
				avec le clos de la conclusion

	_clos_infreq <set> :
		liste des clos avant abstraction qui deviennent infrequents apres abstraction
	
	_minsup <int> :
		Support minimal d'un clos pour etre considere comme frequent apres abstraction
	
	_mindeg <int> :
		Pour etre conserve lors de l'abstraction, une transaction doit avoir un degree
		minimal superieur ou egal a _mindeg dans le graphe simple des relations

	_mincc <int> :
		Pour etre conserve lors de l'abstraction, une transaction doit appartenir a une
		composante connexe du graphe simple des relations dont la taille est superieure
		ou egale à _mincc

	_mintercc <int> :
		Pour etre conserve lors de l'abstraction, une transaction doit appartenir a une
		composante connexe du graphe des triangles de relations dont la taille est
		superieure ou egale à _mintercc

	"""
	def __init__(self, rules, rel_it, minsup, mindeg, mincc, mintercc, title=None) :
		if not isinstance( rules, ItemsetRules.ItemsetRules) :
			raise AbstractionControllerException( "%s doit être de type ItemsetRules\n"%str(rules))
		self._clos_gen_ori = rules
		
		if not isinstance( rel_it, RelationalItemset.RelationalItemset) :
			raise AbstractionControllerException( "%s doit être de type RelationalItemset\n"%str(rel_it))
		self._rela_itemsets = rel_it
		if not isinstance( minsup, int) :
			raise AbstractionControllerException( "%s doit être de type int\n"%str(minsup))
		self._minsup    = minsup
		if not isinstance( mindeg, int) :
			raise AbstractionControllerException( "%s doit être de type int\n"%str(mindeg))
		self._mindeg    = mindeg
		if not isinstance( mincc, int) :
			raise AbstractionControllerException( "%s doit être de type int\n"%str(mincc))
		self._mincc     = mincc
		if not isinstance( mintercc, int) :
			raise AbstractionControllerException( "%s doit être de type int\n"%str(mintercc))
		self._mintercc  = mintercc
		self._clos_infreq = set([])
		self._title = title
	
	def get_clos_infreq(self) :
		"""
		Retourne la liste des clos_ori devenus infrequents par projection.
		"""
		return self._clos_infreq
	
	def _degree_projection(self, subG) :
		"""
		Applique la projection sur critere de degree:
			[-] supprime du sous graphe subG les noeuds dont le degre est inferieur 
				a mindeg.
			[-] itère la suppression des noeuds de degre insuffisant jusqu'a atteindre
				un point fixe (le graphe contient que des noeuds dont le degre est
				superieur ou egale a _mindeg.
		
		Renvoie le nombre de noeuds conserves dans le graphe apres l'application de la
		contrainte de degre si celui-ci est superieur a _minsup, ou la valeur
		(_minsup - 1) si l'application de la contrainte conduit a un graphe dont le
		nombre de noeuds est inferieur a _minsup. Dans ce dernier cas il se peut que
		le nombre" de noeuds ne soit pas exactement celui qui doit etre obtenu apres
		projection.
		ie: lors de la projection on enleve des noeuds, mais ons'arrette des que l'on passe
		sous _minsup.
		"""
		sup_cur = subG.number_of_nodes()
		if self._mindeg == 0 : return sup_cur
		fixe = False
		while not fixe:
			sup_start = sup_cur
			for node, degree in subG.degree_iter() :
				if degree < self._mindeg :
					subG.remove_node(node)
					sup_cur -= 1
					if sup_cur < self._minsup : return sup_cur 
			fixe = (sup_start == sup_cur )
		return sup_cur
			
	def _cc_projection(self, subG) :
		"""
		Applique la projection sur critere de taille minimale de composante connexe dans le
		graphe simple
			[-] supprime du sous graphe subG les noeuds qui appartiennent a une
				composante connexe du graphe simple des relation dont la taille est 
				strictement inferieur a _mincc.
		
		Renvoie le nombre de noeuds conserves dans le graphe apres l'application de la
		contrainte de taille minimale de composante connexe  si celui-ci est superieur
		a _minsup, ou un valeur inferieure a _minsup si l'application de la contrainte
		conduit a un graphe dont le nombre de noeuds est inferieur a _minsup. Dans ce dernier
		cas il se peut que le nombre" de noeuds ne soit pas exactement celui qui doit
		etre obtenu apres projection.
		ie: lors de la projection on enleve des noeuds, mais ons'arrette des que l'on passe
		sous _minsup.
		"""
		sup_cur = subG.number_of_nodes()
		if self._mincc == 0 : return sup_cur
		cc = nx.connected_components(subG)
		for c in cc :
			if len(c) < self._mincc:
				subG.remove_nodes_from(c)
				sup_cur -= len(c)
				if sup_cur < self._minsup : return sup_cur 
		return sup_cur
	
	def _tercc_projection(self, subG) :
		"""
		Applique la projection sur un critere de taille minimal de composante connexe dans le
		graphe des triangles.
			[-] supprime du sous graphe subG les noeuds qui n'apparaissent pas dans une
			composante connexe du graphe des triangles dont la taille est strictement
			inferieur a _mintercc

		Renvoie le nombre de noeuds conserves dans le graphe apres l'application de la
		contrainte de taille minimale de composante connexe  si celui-ci est superieur
		a _minsup, ou un valeur inferieure a _minsup si l'application de la contrainte
		conduit a un graphe dont le nombre de noeuds est inferieur a _minsup. Dans ce dernier
		cas il se peut que le nombre" de noeuds ne soit pas exactement celui qui doit
		etre obtenu apres projection.
		ie: lors de la projection on enleve des noeuds, mais ons'arrette des que l'on passe
		sous _minsup.
		"""
		sup_cur = subG.number_of_nodes()
		if self._mintercc == 0 : return sup_cur
		in_ter_cc = GrapheDesTriangles.GrapheDesTriangles(subG).nodes_in_cc_of_size_at_least(self._mintercc)
		for node in subG.nodes() :
			if node not in in_ter_cc :
				subG.remove_node(node)
				sup_cur -= 1
				if sup_cur < self._minsup : return sup_cur
		return sup_cur
	
	def _projection( self, clos) :
		"""
		Efectue la projection du clos sur le graphe en fonction des criteres choisis lors
		de l'instanciation du controller d'abstraction.
		La methode renoive le sous graphe resultat de la projection du clos.
		
		Attention: Si la projection conduit a un sous-graphe dont le nombre de noeuds
		est inferieur au _minsup, alors le sous-graphe renvoye peut ne pas etre le resultat
		de la projection
		ie: lors de la projection on enleve des noeuds, mais ons'arrette des que l'on passe
		sous _minsup.
		"""
		emptyset = Itemset.Itemset(0,self._rela_itemsets.get_items())
		if emptyset == clos :
			subG = nx.Graph(self._rela_itemsets.get_graph())
		else :
			subG = self._rela_itemsets.subgraph(clos)
		# on applique les abstraction jusqu'a ce qu'on atteigne un point fixe
		# (l'extension ne change plus si on applique la (les) projections ou
		# que le support soit inferieur a minsup
		fixe = False
		sup_cur = subG.number_of_nodes()
		while not fixe:
			sup_start = sup_cur
			# application de la projection sur critere de degre
			sup_cur = self._degree_projection(subG)
			if sup_cur < self._minsup : break
			# Verifie si on est a un point fixe.
			fixe = ( sup_start == sup_cur )
		# application de la projection sur critere de taille de composante
		# connexe dans le graphe simple
		sup_cur = self._cc_projection(subG)
		if sup_cur < self._minsup : return subG
		# application de la projection sur critere de taille de composante
		# connexe dans le graphe des triangles
		sup_cur = self._tercc_projection(subG)
		return subG
	
	def reduce_to_most_general(self, liste_ori):
		"""
		Permet de reduire une liste de motif a l'ensemble des ses composants les plus
		generaux. Autrement dit, si un motif de la liste passee entree est une
		specialisation d'un autre element de la liste, il est retire.
		"""
		more_gen = set([])
		for g in liste_ori :
			is_not_the_more_general = False
			for go in liste_ori :
				is_not_the_more_general = ( go != g and  go.intersection(g) == go )
				if is_not_the_more_general : break
			if not is_not_the_more_general : more_gen.add( g )
		return more_gen
	
		

class GlobalAbstractionController (AbstractionController) :
	"""
	Effectue l'abstraction locale et calcule les nouvelles regles abstraites.
	"""
	def __init__(self, rules, rel_it, minsup, mindeg, mincc, mintercc, title=None) :
		AbstractionController.__init__(self,rules, rel_it, minsup, mindeg, mincc, mintercc, title=title)
		self._fusion_clos  = ItemsetRules.ItemsetRules( self._rela_itemsets.get_items() )
		self._clos_gen_abs = ItemsetRules.ItemsetRules( self._rela_itemsets.get_items() )
		self._correlation_structurale = dict()

	
	def get_fusion_clos(self) :
		"""
		Retourne les regles clos_abs <- clos_ori
		"""
		return self._fusion_clos
	
	def get_clos_gen_abs(self) :
		"""
		Retourne les regles clos_abs <- generateurs_abs ...
		"""
		return self._clos_gen_abs
	
	def get_correlation_structurale(self):
		"""
		Retourne le dictionnaire donnant la correlation structturale (val) de chaque clos
		abstrait (key).
		"""
		return self._correlation_structurale
	
	def _add_correlation_structurale( self, clos_abs, sup_ori, sup_abs) :
		if sup_ori == 0 :
			self._correlation_structurale[clos_abs] = None
		else :
			self._correlation_structurale[clos_abs] = float(sup_abs) / float(sup_ori)
	
	def abstract( self) :
		"""
		Realise l'abstraction de tous les clos.
		"""
		# projection de chaque clos
		for clos_ori, (gen_ori,sup_ori) in self._clos_gen_ori.iter_rules() :
			# Calcul de l'extension abstraite du clos.
			subG = self._projection( clos_ori)
			# Si on est sortie avec un point fixe (et donc non pour cause d'infrequence
			# alors on traite le clos abstrait
			if subG.number_of_nodes() < self._minsup :
				self._clos_infreq.add(clos_ori)
				continue # --> passe au clos suivant car celui la est infrequent par proj.
			else :
				ext_abs = subG.nodes()[:]
				subG=None
				# calcul du cloture de l'extension abstraite
				clos_abs = self._rela_itemsets.cloture( *ext_abs )
				# ajoute la regle clos_abs <- clos_ori
				self._fusion_clos.add_rule(clos_abs, clos_ori)
				self._fusion_clos.set_extension( clos_abs, ext_abs)
				# ajoute une regle aux regles clos_abs <- gen_i
				# temporairement gen_i est ici un ensemble de generateur non-abstrait.
				# leur reduction aux generateurs abstraits se fait dans une second temps
				self._clos_gen_abs.add_rule(clos_abs, *gen_ori)
				self._clos_gen_abs.set_extension(clos_abs, len(ext_abs))
				# memorise la correlation structurale du clos abstrait
				self._add_correlation_structurale( clos_abs, len(self._rela_itemsets.extension(clos_abs)), len(ext_abs))
		# Lors de la fusion des classes d'quivalence et du calcul des clos abstraits, les
		# generateurs abstraits sont caclules a partir de la liste des generateurs des clos
		# originaux donnant par abstraction le meme clos abstrait. Cette methode calcul
		# les generateurs abstraits a partir des generateurs originaux.
		for clos_abs, (gen_ori,ext) in self._clos_gen_abs.iter_rules() :
			self._clos_gen_abs.set_premisses(clos_abs, self.reduce_to_most_general(gen_ori))
	
	def __str__(self) :
		s  = self._title + "\n\n"
		s += "# clos_gen_abs ------------------\n"
		s += str(self.get_clos_gen_abs())+"\n\n"
		s += "# fusion_clos  ------------------\n"
		s += str(self.get_fusion_clos())+"\n\n"
		s += "# clos_infreq  ------------------\n"
		s += string.join(map(lambda it: str(it),self.get_clos_infreq()),',')+"\n\n"
		s += "# correlation structurale -------\n"
		for it, cs in sorted( self.get_correlation_structurale().iteritems(), key=lambda i:i[1], reverse=True):
			s += str(it) + " -> " + str(cs) + "\n"
		return s


class LocalAbstractionController (AbstractionController) :
	"""
	Effectue l'abstraction locale et calcule les nouvelles regles abstraites.
			[-] si le controller est instancie avec _in_simple = True, alors les composantes
			connexes sont calculees dans le graphe simple (direct).
			[-] sinon, les comp. conn. sont calculees dans le graphe des triangles.
	"""
	def __init__(self, rules, rel_it, minsup, mindeg, mincc, mintercc, title=None, in_simple = True) :
		AbstractionController.__init__(self,rules, rel_it, minsup, mindeg, mincc, mintercc,title=title)
		self._clos_loc_desc = dict()
		self._in_simple = in_simple
		self._red_rules_and_spec = dict()
		
	def get_clos_loc_desc(self) :
		"""
		Retourne la description des clos locaux sous la forme:
		
		(clos_loc_abs, ext, clos_ori)

		Avec:
			[-] clos_loc_abs	: le clos local abstrait.
			[-] ext				:l'extension du clos local (ie: la description de la localite).
			[-] clos_ori		: le clos originel dont il derive.

		"""
		return self._clos_loc_desc
	
	def connected_components( self, subG) :
		"""
		Renvoie les composantes connexe sous forme de liste de noeuds simples, que les comp.
		conn. soient calculees dans le graphe simple ou dans le graphe des triangles.
		"""
		# Calcul des comp. con. dans lesquelles les clotures sont effectuees
		if self._in_simple :
			# les composantes connexes sont calculées dans le graphe simple
			return [ c for c in nx.connected_components(subG) if len(c) >= self._minsup and len(c) >= self._mincc ]
		else :
			# les composantes connexes sont calculées dans le graphe des triangles
			return [c for c in map(lambda cc : set().union(*cc), nx.connected_components(GrapheDesTriangles.GrapheDesTriangles(subG)
)) if len(c) >= self._minsup and len(c) >= self._mintercc]
		
	def abstract( self) :
		"""
		Realise l'abstraction de tous les clos dans une version locale.
		ie. les clos sont calcules dans chaque composante connexe de facon independant.
			[-] si le controller est instancie avec _in_simple = True, alors les composantes
			connexes sont calculees dans le graphe simple (direct).
			[-] sinon, les comp. conn. sont calculees dans le graphe des triangles.
		"""
		# projection de chaque clos
		for clos_ori, (gen_ori,sup_ori) in self._clos_gen_ori.iter_rules() :
			# Calcul de l'extension abstraite du clos.
			subG = self._projection( clos_ori)
			# Si on est sortie avec un point fixe (et donc non pour cause d'infrequence
			# alors on traite le clos abstrait
			if subG.number_of_nodes() < self._minsup :
				self._clos_infreq.add(clos_ori)
				continue # --> passe au clos suivant car celui la est infrequent par proj.
			else :
				# Calcul des comp. con. dans lesquelles les clotures sont effectuees
				cc = self.connected_components( subG)
				subG=None
				if cc == [] :
					self._clos_infreq.add(clos_ori)
					continue
				for ext_loc_abs in cc :
					ext_loc_abs = tuple(sorted(list(ext_loc_abs)))
					# calcul du cloture de l'extension abstraite locale
					clos_loc_abs = self._rela_itemsets.cloture( *ext_loc_abs )
					# ajoute le triplet (clos_loc_abs, ext_loc, clos_ori) 
					try :
						self._clos_loc_desc[ext_loc_abs][1].add(clos_ori)
					except KeyError :
						self._clos_loc_desc[ext_loc_abs] = [clos_loc_abs, set([clos_ori])]
		# Calcul de l'indicateur de specificite des clos abstraits et reduction
		# de l'ensemble des regles par suppression des clos_ori les moins généraux
		for ext, (clos_abs, clos_ori) in self._clos_loc_desc.iteritems() : 
			self._red_rules_and_spec[ext]= ( clos_abs, self.reduce_to_most_general(list(clos_ori)), len(ext)/float(self._rela_itemsets.support(clos_abs)))
	
	def __str__(self):
		s  = self._title + "\n\n"
		s += "# clos_loc_abs ------------------\n"
		for ext, (clos_abs, liste_clos_ori) in sorted(self.get_clos_loc_desc().items(), key=lambda i: i[1][0], reverse=True) :
			s += str(clos_abs) + " (" + str(ext) + ") +; [" + string.join(map(lambda it: str(it), liste_clos_ori), ", ")+ "]\n"
		s += "\n# clos_infreq  ------------------\n"
		s += string.join(map(lambda it: str(it),self.get_clos_infreq()),', ')
		s += "\n\n# reduced_rules -----------------\n"
		for ext, (clos_abs, clos_red, specif) in self._red_rules_and_spec.iteritems():
			s += str(specif) + " " + str(ext ) + " "+str(clos_abs) + " [" + string.join(map(lambda it: str(it), clos_red), ", ") +"]\n"
		return s

		


if __name__ == '__main__' :
	#-----------------------------------------------
	# Chargement de l'itemset relationnel
	from RIParser import RIParser
	fi = open('data/mougel.ri', 'r')
	parser = RIParser(fi)
	ri = parser.parse()
	fi.close()
	#-----------------------------------------------
	# Chargement du fichier de regles
	from FCIGEParser import FCIGEParser
	fi = open( 'data/mougel_0.fcige', 'r')
	parser = FCIGEParser( fi, ri.get_items())
	clos_gen = parser.parse()
	fi.close()
	#-----------------------------------------------
	MIN_SUPP   = 0
	MIN_DEG    = 0
	MIN_CC     = 0
	MIN_TERCC  = 4
	#-----------------------------------------------
	#abstractor = GlobalAbstractionController( clos_gen, ri, MIN_SUPP, MIN_DEG, MIN_CC, MIN_TERCC, title="Test Global")
	#abstractor.abstract()
	#print abstractor
	#-----------------------------------------------
	print
	print
	IN_SIMPLE = True
	abstractor =  LocalAbstractionController( clos_gen, ri, MIN_SUPP, MIN_DEG, MIN_CC, MIN_TERCC, title="Test Local", in_simple=IN_SIMPLE)
	abstractor.abstract()
	print(abstractor)
